﻿// --------------------------------------------------------------------------
// Project Frocessing
// ActionScript 3.0 drawing library like Processing.
// --------------------------------------------------------------------------
//
// Frocessing drawing library
// Copyright (C) 2008-09  TAKANAWA Tomoaki (http://nutsu.com) and
//					   	  Spark project (www.libspark.org)
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// contact : face(at)nutsu.com
//

package frocessing.core {
	
	import flash.display.Graphics;
	import flash.display.BitmapData;
	import flash.geom.Matrix;
	import frocessing.geom.FNumber3D;
	import frocessing.core.render3d.*;
	//import frocessing.f3d.lights.*;
	import frocessing.bmp.FBitmapData;
	
	/**
	* GraphicsEx3D
	* 
	* @author nutsu
	* @version 0.5
	*/
	public class GraphicsEx3D extends GraphicsEx
	{
		//constants
		private static const TASK_SHAPE:int   = 1;
		private static const TASK_STROKE:int  = 2;
		private static const TASK_POINT:int   = 10;
		private static const TASK_PIXEL:int   = 11;
		private static const TASK_IMAGE:int   = 20;
		private static const TASK_POLYGON:int = 30;
		private static const TASK_POLYGON_SOLID:int = 31;
		
		private static const FILL_NO:int       = 0;
		private static const FILL_SOLID:int    = 1;
		private static const FILL_BITMAP:int   = 2;
		private static const FILL_GRADIENT:int = 3;
		
		//center coordinates
		public var centerX:Number;
		public var centerY:Number;
		
		//DRAW PATH TASK
		private var TASK_DAT:Array;
		private var task_count:uint;
		private var shape_task:RenderTask;
		private var STROKE_TASK:Array;	//IStrokeTask[]
		private var stroke_count:uint;
		private var current_stroke_index:int;
		private var current_fill_type:int;
		
		//
		private var fill_matrix:Matrix;
		private var fill_repeat:Boolean;
		private var fill_smooth:Boolean;
		private var grad_type:String;
		private var grad_colors:Array;
		private var grad_alphas:Array;
		private var grad_ratios:Array
		private var grad_spreadMethod:String;
		private var grad_interpolationMethod:String;
		private var grad_focalPointRation:Number;
		
		//PATH
		private var path_start_index:int;
		private var cmd_start_index:int;
		
		private var paths:Array;
		private var path_count:int;
		private var vertex_count:int;
		
		private var commands:Array;
		private var cmd_count:int;
		
		private var do_shape_render:Boolean;
		private var path_z_sum:Number;
		private var z_out:Boolean;
		
		//
		private var stroke_only:Boolean;
		private var stroke_applyed:Boolean = false;
		private var fill_applyed:Boolean   = false;
		
		//
		private var add_shape_task:Boolean = false;
		
		//
		public var perspective:Boolean;
		public var zNear:Number = 100;
		
		//
		public var backFaceCulling:Boolean;
		
		private var __texture:BitmapData;
		private var __texture_back:BitmapData;
		
		
		//path status
		/**
		 * @private
		 */
		internal var __startZ:Number = 0;
		/**
		 * @private
		 */
		internal var __z:Number = 0;
		
		/**
		 * 
		 */
		public function GraphicsEx3D( graphics:Graphics, centerX:Number=0, centerY:Number=0, imageSmoothing:Boolean=false ) 
		{
			super(graphics);
			
			//
			this.imageSmoothing = imageSmoothing;
			imageDetail = 4;
			
			//texture
			__texture   	= null;
			__texture_back 	= null;
			backFaceCulling = true;
			
			//
			fillDo           = true;
			__stroke         = true;
			
			perspective  	= true;
			this.centerX 	= centerX;
			this.centerY 	= centerY;
			
			defaultSetting();
		}
		
		public function defaultSetting():void
		{
			TASK_DAT         = [];
			task_count       = 0;
			
			STROKE_TASK      = [];
			stroke_count     = 0;
			STROKE_TASK[stroke_count] = new NoStrokeTask();
			stroke_count++;
			
			stroke_only      = false;
			stroke_applyed   = __stroke;
			fill_applyed	 = false;
			
			current_fill_type = FILL_NO;
			
			// path init
			path_start_index = 0;
			cmd_start_index  = 0;
			
			paths            = [];
			path_count       = 0;
			commands         = [];
			cmd_count        = 0;
			
			do_shape_render  = false;
			vertex_count     = 0;
			path_z_sum       = 0;
			
			add_shape_task   = false;
			
			//
			__startX = __startY = __startZ = 0;
			__x = __y = __z = 0;
		}
		
		//-------------------------------------------------------------------------------------------------------------------
		// 
		//-------------------------------------------------------------------------------------------------------------------
		
		/**
		 * 
		 */
		public function beginDraw( perspective:Boolean=true ):void
		{
			this.perspective = perspective;
			defaultSetting();
		}
		
		/**
		 * 
		 */
		public function endDraw():void
		{
			render();
			__texture = null;
			__texture_back = null;
		}
		
		/**
		 * @inheritDoc
		 */
		override public function clear():void
		{
			gc.clear();
			defaultSetting();
		}
		
		//------------------------------------------------------------------------------------------------------------------- override
		
		/**
		 * @private
		 */
		override internal function __reset():void
		{
			;//nothing
		}
		
		override public function moveTo( x:Number, y:Number ):void
		{
			moveTo3d( x, y, 0 );
		}
		override public function lineTo( x:Number, y:Number ):void
		{
			lineTo3d( x, y, 0 );
		}
		override public function curveTo( cx:Number, cy:Number, x:Number, y:Number ):void
		{
			curveTo3d( cx, cy, 0, x, y, 0 );
		}
		override public function bezierTo( cx0:Number, cy0:Number, cx1:Number, cy1:Number, x:Number, y:Number ):void
		{
			bezierTo3d( cx0, cy0, 0, cx1, cy1, 0, x, y, 0 );
		}
		override public function splineTo( cx0:Number, cy0:Number, x:Number, y:Number, cx1:Number, cy1:Number ):void
		{
			splineTo3d( cx0, cy0, 0, x, y, 0, cx1, cy1, 0 );
		}
		override public function point(x:Number, y:Number):void 
		{
			point3d( x, y, 0 );
		}
		
		/**
		 * @inheritDoc
		 */
		override public function moveToLast():void
		{
			var x0:Number = __x;
			var y0:Number = __y;
			if ( perspective )
			{
				x0 *= __z/zNear;
				y0 *= __z/zNear;
			}
			moveTo3d( x0, y0, __z );
		}
		
		//------------------------------------------------------------------------------------------------------------------- 3D Path
		
		/**
		 * 
		 */
		public function bezierTo3d( cx0:Number, cy0:Number, cz0:Number, cx1:Number, cy1:Number, cz1:Number, x:Number, y:Number, z:Number ):void
		{
			var x0:Number = __x;
			var y0:Number = __y;
			var z0:Number = __z;
			if ( perspective )
			{
				x0 *= z0/zNear;
				y0 *= z0/zNear;
			}
			var k:Number = 1.0/bezierDetail;
			var t:Number = 0;
			var tp:Number;
			for ( var i:int = 1; i <= bezierDetail; i++ )
			{
				t += k;
				tp = 1.0 - t;
				lineTo3d( x0*tp*tp*tp + 3*cx0*t*tp*tp + 3*cx1*t*t*tp + x*t*t*t,
						  y0*tp*tp*tp + 3*cy0*t*tp*tp + 3*cy1*t*t*tp + y*t*t*t, 
						  z0*tp*tp*tp + 3*cz0*t*tp*tp + 3*cz1*t*t*tp + z*t*t*t );
			}
		}
		
		/**
		 * 
		 */
		public function splineTo3d( cx0:Number, cy0:Number, cz0:Number, x:Number, y:Number, z:Number, cx1:Number, cy1:Number, cz1:Number ):void
		{
			var x0:Number = __x;
			var y0:Number = __y;
			var z0:Number = __z;
			if ( perspective )
			{
				x0 *= z0/zNear;
				y0 *= z0/zNear;
			}
			var k:Number = 1.0 / splineDetail;
			var t:Number = 0;
			//convert to bezier
			var cx0:Number = x0 + __tightness * ( x - cx0 );
			var cy0:Number = y0 + __tightness * ( y - cy0 );
			var cz0:Number = z0 + __tightness * ( z - cz0 );
			var cx1:Number = x  - __tightness * ( cx1 - x0 );
			var cy1:Number = y  - __tightness * ( cy1 - y0 );
			var cz1:Number = z  - __tightness * ( cz1 - z0 );
			var tp:Number;
			for ( var i:int = 1; i <= splineDetail; i++ )
			{
				t += k;
				tp = 1.0-t;
				lineTo3d( x0*tp*tp*tp + 3*cx0*t*tp*tp + 3*cx1*t*t*tp + x*t*t*t,
						  y0*tp*tp*tp + 3*cy0*t*tp*tp + 3*cy1*t*t*tp + y*t*t*t, 
						  z0*tp*tp*tp + 3*cz0*t*tp*tp + 3*cz1*t*t*tp + z*t*t*t );
			}
		}
		
		//------------------------------------------------------------------------------------------------------------------- 
		// STYLE
		//-------------------------------------------------------------------------------------------------------------------
		
		/**
		 * @inheritDoc
		 */
		override public function applyStroke():void
		{
			__stroke = true;
			stroke_applyed = true;
		}
		
		/**
		 * @inheritDoc
		 */
		override public function noStroke():void
		{
			__stroke = false;
		}
		
		/**
		 * @inheritDoc
		 */
		override public function applyFill():void
		{
			if ( fillDo )
			{
				beginFill( fillColor, fillAlpha );
			}
		}
		
		/**
		 * not implemented
		 */
		override public function lineGradientStyle( type:String, colors:Array, alphas:Array, ratios:Array,
										   matrix:Matrix = null, spreadMethod:String = "pad", interpolationMethod:String = "rgb",
										   focalPointRatio:Number=0.0 ):void
		{
			;
		}
		
		//------------------------------------------------------------------------------------------------------------------- Fill
		
		/**
		 * @inheritDoc
		 */
		override public function beginFill(color:uint, alpha:Number=1.0):void
		{
			fillColor = color;
			fillAlpha = alpha;
			fillDo = true;
			fill_applyed = true;
			current_fill_type = FILL_SOLID;
			
			// ------------ START PATH ------------
			terminateShapePath();
			startShapePath();
		}
		
		/**
		 * @inheritDoc
		 */
		override public function beginBitmapFill( bitmapdata:BitmapData, matrix:Matrix=null, repeat:Boolean=true, smooth:Boolean=false ):void
		{
			__texture   = bitmapdata;
			fill_matrix = ( matrix==null ) ? null : matrix.clone();
			fill_repeat = repeat;
			fill_smooth = smooth;
			fillDo = true;
			fill_applyed = true;
			current_fill_type = FILL_BITMAP;
			
			// ------------ START PATH ------------
			terminateShapePath();
			startShapePath();
		}
		
		/**
		 * @inheritDoc
		 */
		override public function beginGradientFill( type:String, colors:Array, alphas:Array, ratios:Array,
										   matrix:Matrix = null, spreadMethod:String = "pad", interpolationMethod:String = "rgb",
										   focalPointRation:Number=0.0 ):void
		{
			grad_type   = type;
			grad_colors = colors;
			grad_alphas = alphas;
			grad_ratios = ratios;
			fill_matrix = ( matrix==null ) ? null : matrix.clone();
			grad_spreadMethod = spreadMethod;
			grad_interpolationMethod = interpolationMethod;
			grad_focalPointRation = focalPointRation;
			fillDo = true;
			fill_applyed = true;
			current_fill_type = FILL_GRADIENT;
			
			// ------------ START PATH ------------
			terminateShapePath();
			startShapePath();
		}
		
		/**
		 * @inheritDoc
		 */
		override public function endFill():void
		{
			if ( fill_applyed )
			{
				terminateShapePath();
				current_fill_type = FILL_NO;
			}
			fill_applyed = false;
		}
		
		public function beginTexture( texture:BitmapData, back_texture:BitmapData=null ):void
		{
			__texture = texture;
			__texture_back = (back_texture!=null) ? back_texture : texture;
		}
		
		public function endTexture():void
		{
			__texture = null;
			__texture_back = null;
		}
		
		//------------------------------------------------------------------------------------------------------------------- 
		// buffer
		//-------------------------------------------------------------------------------------------------------------------
		
		/**
		 * add stroke
		 */
		private function addStrokeTask():void
		{
			if ( stroke_applyed )
			{
				STROKE_TASK[stroke_count] = new StrokeTask( thickness, strokeColor, strokeAlpha, pixelHinting, scaleMode, caps, joints, miterLimit );
				stroke_count++;
				stroke_applyed = false;
			}
		}
		
		/**
		 * start shape draw session. moveTo, beginFill, beginBitmapFill, beginGradientFill
		 * @private
		 */
		private function startShapePath():void
		{
			//start indexes
			path_start_index = path_count;
			cmd_start_index  = cmd_count;
			
			//current shape parameters
			do_shape_render = true;
			vertex_count = 0;
			path_z_sum   = 0;
			
			//stroke
			if ( __stroke )
			{
				addStrokeTask();
				current_stroke_index = stroke_count - 1;
			}
			else
			{
				current_stroke_index = 0;
			}
		}
		
		/**
		 * terminate shape draw session.
		 * @private
		 */
		private function terminateShapePath():void
		{
			if ( add_shape_task )
			{
				if ( do_shape_render || stroke_only )	//all z coordinates < 0
				{
					switch( current_fill_type )
					{
						case FILL_SOLID:
							shape_task = new RenderTask( TASK_SHAPE, path_start_index, cmd_start_index, path_z_sum/vertex_count, cmd_count - cmd_start_index );
							shape_task.fillColor = fillColor;
							shape_task.fillAlpha = fillAlpha;
							shape_task.fillDo = true;
							break;
						case FILL_BITMAP:
							shape_task = new RenderBitmapTask( TASK_SHAPE, path_start_index, cmd_start_index, path_z_sum / vertex_count, cmd_count - cmd_start_index );
							RenderBitmapTask(shape_task).setParameters( __texture, fill_matrix, fill_repeat, fill_smooth );
							shape_task.fillDo = true;
							break;
						case FILL_GRADIENT:
							shape_task = new RenderGradientTask( TASK_SHAPE, path_start_index, cmd_start_index, path_z_sum / vertex_count, cmd_count - cmd_start_index );
							RenderGradientTask(shape_task).setParameters( grad_type, grad_colors, grad_alphas, grad_ratios, fill_matrix,
																		  grad_spreadMethod, grad_interpolationMethod, grad_focalPointRation );
							shape_task.fillDo = true;
							break;
						default: //FILL_NO
							shape_task = new RenderTask( TASK_STROKE, path_start_index, cmd_start_index, path_z_sum/vertex_count, cmd_count - cmd_start_index );
							break;
					}
					shape_task.stroke_index = current_stroke_index;
					TASK_DAT[task_count] = shape_task;
					task_count++;
					stroke_only = false;
				}
				add_shape_task = false;
			}
		}
		
		//------------------------------------------------------------------------------------------------------------------- 
		// Shape TASK
		//-------------------------------------------------------------------------------------------------------------------
		
		/**
		 * 
		 */
		public function moveTo3d( x:Number, y:Number, z:Number ):void
		{
			// not start shape session, while fill continue.
			if ( !fill_applyed )
			{
				current_fill_type = FILL_NO;
				// ------------ START TASK_STROKE ------------
				terminateShapePath();
				startShapePath();
			}
			
			// check out of render
			z_out = ( z <= 0 );
			
			if ( perspective )
			{	
				x *= zNear/z;
				y *= zNear/z;
			}
			
			// path coordinates
			__startX = __x = x;
			__startY = __y = y;
			__startZ = __z = z;
			paths[ path_count ] = __startX + centerX;
			path_count++;
			paths[ path_count ] = __startY + centerY;
			path_count++;
			paths[ path_count ] = __startZ;
			path_count++;
			
			// commands
			commands[cmd_count] = 1; //MOVE_TO
			
			// current shape parameter
			vertex_count++;
			path_z_sum += z;
			if( z_out ) do_shape_render = false;
			
			// global draw command count
			cmd_count++;
			//current_stroke_index = 0;
		}
		
		/**
		 * 
		 */
		public function lineTo3d( x:Number, y:Number, z:Number ):void
		{
			// check out of render
			z_out = ( z <= 0 );
			
			if ( perspective )
			{
				x *= zNear/z;
				y *= zNear/z;
			}
			
			// path coordinates
			__x = x;
			__y = y;
			__z = z;
			paths[ path_count ] = x + centerX;
			path_count++;
			paths[ path_count ] = y + centerY;
			path_count++;
			paths[ path_count ] = z;
			path_count++;
			
			// commands
			commands[cmd_count] = 2; //LINE_TO
			
			// current shape parameter
			vertex_count++;
			path_z_sum += z;
			if ( z_out ) do_shape_render = false;
			
			// global draw command count
			cmd_count++;
			
			//add flag
			add_shape_task = true;
		}
		
		/**
		 * 
		 */
		public function curveTo3d( cx:Number, cy:Number, cz:Number, x:Number, y:Number, z:Number ):void
		{
			// check out of render
			z_out = ( z <= 0 );
			
			if ( perspective )
			{
				cx *= zNear/cz;
				cy *= zNear/cz;
				x  *= zNear/z;
				y  *= zNear/z;
			}
			
			// path coordinates
			__x = x;
			__y = y;
			__z = z;
			paths.push( cx + centerX, cy + centerY, cz,
						x + centerX,  y + centerY,  z );
			path_count += 6;
			
			// commands
			commands[cmd_count] = 3; //CURVE_TO
			
			// current shape parameter
			vertex_count++;
			path_z_sum += z;
			if ( z_out ) do_shape_render = false;
			
			// global draw command count
			cmd_count++;
			
			//add flag
			add_shape_task = true;
		}
		
		/**
		 * 
		 */
		override public function closePath():void
		{
			// path coordinates
			__x = __startX;
			__y = __startY;
			__z = __startZ;
			paths[ path_count ] = __startX + centerX;
			path_count++;
			paths[ path_count ] = __startY + centerY;
			path_count++;
			paths[ path_count ] = __startZ;
			path_count++;
			
			// commands
			commands[cmd_count] = 2; //LINE_TO
			
			// global draw command count
			cmd_count++;
		}
		
		//------------------------------------------------------------------------------------------------------------------- 
		// Other TASK
		//-------------------------------------------------------------------------------------------------------------------
		
		/**
		 * 
		 */
		public function point3d( x:Number, y:Number, z:Number ):void
		{
			terminateShapePath();
			
			// check out of render
			z_out = ( z <= 0 );
			
			if ( !z_out )
			{
				if ( perspective )
				{
					x *= zNear/z;
					y *= zNear/z;
				}
				__x = __startX = x;
				__y = __startY = y;
				__z = __startZ = z;
				
				// add task
				var task:PointTask = new PointTask( x + centerX, y + centerY, z, strokeColor, strokeAlpha );
				TASK_DAT[task_count] = task;
				task_count++;
			}
		}
		
		/**
		 * 
		 */
		public function pixel3d( x:Number, y:Number, z:Number ):void
		{
			terminateShapePath();
			
			// check out of render
			z_out = ( z <= 0 );
			
			if ( !z_out )
			{
				if ( perspective )
				{
					x *= zNear/z;
					y *= zNear/z;
				}
				__x = __startX = x;
				__y = __startY = y;
				__z = __startZ = z;
				
				// add task
				var task:PixelTask = new PixelTask( x + centerX, y + centerY, z, uint(strokeAlpha*0xff)<<24 | strokeColor  );
				TASK_DAT[task_count] = task;
				task_count++;
			}
		}
		
		//-------------------------------------------------------------------------------------------------------------------
		
		/**
		 * 
		 */
		public function bitmap( x:Number, y:Number, z:Number, w:Number, h:Number, center:Boolean ):void
		{
			terminateShapePath();
			
			// check out of render
			z_out = ( z <= 0 );
			
			__x = __startX = x;
			__y = __startY = y;
			__z = __startZ = z;
			
			if ( !z_out )
			{
				if ( perspective )
				{
					x *= zNear/z;
					y *= zNear/z;
					w *= zNear/z;
					h *= zNear/z;
				}
				if ( center )
				{
					x -= w * 0.5;
					y -= h * 0.5;
				}
				
				// add task
				var task_i:ImageTask = new ImageTask( __texture, x + centerX, y + centerY, z, w, h );
				if ( __stroke )
				{
					addStrokeTask();
					task_i.stroke_index = stroke_count-1;
				}
				TASK_DAT[task_count] = task_i;
				task_count++;
			}
		}
		
		/**
		 * 
		 */
		public function polygonSolid( x0:Number, y0:Number, z0:Number, x1:Number, y1:Number, z1:Number, x2:Number, y2:Number, z2:Number ):void
		{
			terminateShapePath();
			
			// check out of render
			z_out = ( z0 <= 0 || z1 <= 0 || z2 <= 0 );
			
			__x = __startX = x0;
			__y = __startY = y0;
			__z = __startZ = z0;
			
			if ( !z_out )
			{
				if ( perspective )
				{
					x0 *= zNear/z0;
					y0 *= zNear/z0;
					x1 *= zNear/z1;
					y1 *= zNear/z1;
					x2 *= zNear/z2;
					y2 *= zNear/z2;
				}
				
				var v0x:Number = x1 - x0; 
				var v0y:Number = y1 - y0;
				var v1x:Number = x2 - x0;
				var v1y:Number = y2 - y0;
				var cross_value:Number = v0x * v1y - v0y * v1x;
				if ( backFaceCulling && cross_value<=0)
					return;
				
				//light test
				//var fillColor_:uint;
				//light.fillColor = fillColor;
				//fillColor_ = light.getShadedColor( v0x, v0y, z1 - z0, v1x, v1y, z2 - z0 );
				
				//
				x0 += centerX;
				y0 += centerY;
				x1 += centerX;
				y1 += centerY;
				x2 += centerX;
				y2 += centerY;
				
				// add task
				var task:PolygonTask;
				switch( current_fill_type )
				{
					case FILL_SOLID:
						task = new PolygonColorTask( x0, y0, z0, x1, y1, z1, x2, y2, z2 );
						PolygonColorTask( task ).setParameters( fillColor, fillAlpha );
						break;
					case FILL_BITMAP:
						task = new PolygonBitmapTask( x0, y0, z0, x1, y1, z1, x2, y2, z2 );
						PolygonBitmapTask(task).setParameters( __texture, fill_matrix, fill_repeat, fill_smooth );
						break;
					case FILL_GRADIENT:
						task = new PolygonGradientTask( x0, y0, z0, x1, y1, z1, x2, y2, z2 );
						PolygonGradientTask(task).setParameters( grad_type, grad_colors, grad_alphas, grad_ratios, fill_matrix,
																 grad_spreadMethod, grad_interpolationMethod, grad_focalPointRation );
						break;
					default:
						task = new PolygonTask( x0, y0, z0, x1, y1, z1, x2, y2, z2 );
						break;
				}
				if ( __stroke )
				{
					addStrokeTask();
					task.stroke_index = stroke_count-1;
				}
				TASK_DAT[task_count] = task;
				task_count++;
			}
		}
		
		/**
		 * 
		 */
		public function plane( x0:Number, y0:Number, z0:Number, x1:Number, y1:Number, z1:Number, x2:Number, y2:Number, z2:Number, x3:Number, y3:Number, z3:Number ):void
		{
			terminateShapePath();
			
			// check out of render
			z_out = ( z0 <= 0 || z1 <= 0 || z2 <= 0 || z3 <= 0 );
			
			__x = __startX = x0;
			__y = __startY = y0;
			__z = __startZ = z0;
			
			if ( !z_out )
			{
				if ( perspective )
				{
					x0 *= zNear/z0;
					y0 *= zNear/z0;
					x1 *= zNear/z1;
					y1 *= zNear/z1;
					x2 *= zNear/z2;
					y2 *= zNear/z2;
					x3 *= zNear/z3;
					y3 *= zNear/z3;
				}
				
				var v0x:Number = x1 - x0; 
				var v0y:Number = y1 - y0;
				var v1x:Number = x2 - x0;
				var v1y:Number = y2 - y0;
				var cross_value:Number = v0x * v1y - v0y * v1x;
				if ( backFaceCulling && cross_value<=0)
					return;
				
				//
				x0 += centerX;
				y0 += centerY;
				x1 += centerX;
				y1 += centerY;
				x2 += centerX;
				y2 += centerY;
				x3 += centerX;
				y3 += centerY;
				
				// add task
				var task:QuadTask;
				switch( current_fill_type )
				{
					case FILL_SOLID:
						task = new QuadColorTask( x0, y0, z0, x1, y1, z1, x2, y2, z2, x3, y3, z3 );
						QuadColorTask( task ).setParameters( fillColor, fillAlpha );
						break;
					case FILL_BITMAP:
						task = new QuadBitmapTask( x0, y0, z0, x1, y1, z1, x2, y2, z2, x3, y3, z3 );
						PolygonBitmapTask(task).setParameters( __texture, fill_matrix, fill_repeat, fill_smooth );
						break;
					case FILL_GRADIENT:
						task = new QuadGradientTask( x0, y0, z0, x1, y1, z1, x2, y2, z2, x3, y3, z3 );
						PolygonGradientTask(task).setParameters( grad_type, grad_colors, grad_alphas, grad_ratios, fill_matrix,
																 grad_spreadMethod, grad_interpolationMethod, grad_focalPointRation );
						break;
					default:
						task = new QuadTask( x0, y0, z0, x1, y1, z1, x2, y2, z2, x3, y3, z3 );
						break;
				}
				if ( __stroke )
				{
					addStrokeTask();
					task.stroke_index = stroke_count-1;
				}
				TASK_DAT[task_count] = task;
				task_count++;
			}
		}
		
		/**
		 * 
		 */
		public function polygon( x0:Number, y0:Number, z0:Number, x1:Number, y1:Number, z1:Number, x2:Number, y2:Number, z2:Number,
								 u0:Number, v0:Number, u1:Number, v1:Number, u2:Number, v2:Number ):void
		{
			terminateShapePath();
			
			// check out of render
			z_out = ( z0 <= 0 || z1 <= 0 || z2 <= 0 );
			
			__x = __startX = x0;
			__y = __startY = y0;
			__z = __startZ = z0;
			
			if ( !z_out )
			{
				if ( perspective )
				{
					x0 *= zNear/z0;
					y0 *= zNear/z0;
					x1 *= zNear/z1;
					y1 *= zNear/z1;
					x2 *= zNear/z2;
					y2 *= zNear/z2;
				}
				
				var v0x:Number = x1 - x0; 
				var v0y:Number = y1 - y0; 
				var v1x:Number = x2 - x0; 
				var v1y:Number = y2 - y0; 
				var cross_value:Number = v0x * v1y - v0y * v1x;
				if ( backFaceCulling && cross_value<=0)
					return;
				
				// add task
				var task:PolygonTextureTask = new PolygonTextureTask(
					x0 + centerX, y0 + centerY, z0,
					x1 + centerX, y1 + centerY, z1,
					x2 + centerX, y2 + centerY, z2,
					u0, v0, u1, v1, u2, v2
				);
				task.bitmapdata = (cross_value>0) ? __texture : __texture_back;
				if ( __stroke )
				{
					addStrokeTask();
					task.stroke_index = stroke_count-1;
				}
				TASK_DAT[task_count] = task;
				task_count++;
			}
		}
		
		//-------------------------------------------------------------------------------------------------------------------
		// 
		//-------------------------------------------------------------------------------------------------------------------
		
		/**
		 * 
		 */
		public function image( x0:Number, y0:Number, z0:Number, x1:Number, y1:Number, z1:Number, x2:Number, y2:Number, z2:Number, x3:Number, y3:Number, z3:Number,
							   u0:Number, v0:Number, u1:Number, v1:Number, u2:Number, v2:Number, u3:Number, v3:Number ):void
		{			
			var u:Number;
			var v:Number;
			
			var v0x0:Number = x0;
			var v0y0:Number = y0;
			var v0z0:Number = z0;
			var v0x1:Number = x1;
			var v0y1:Number = y1;
			var v0z1:Number = z1;
			var v1x0:Number;
			var v1y0:Number;
			var v1z0:Number;
			var v1x1:Number;
			var v1y1:Number;
			var v1z1:Number;
			
			var v0u0:Number = u0;
			var v0v0:Number = u0;
			var v0u1:Number = u1;
			var v0v1:Number = v1;
			var v1u0:Number;
			var v1v0:Number;
			var v1u1:Number;
			var v1v1:Number;
			
			var h0x0:Number; 
			var h0y0:Number;
			var h0z0:Number;
			var h0x1:Number;
			var h0y1:Number;
			var h0z1:Number;
			var h1x0:Number;
			var h1y0:Number;
			var h1z0:Number;
			var h1x1:Number;
			var h1y1:Number;
			var h1z1:Number;
			
			var h0u0:Number;
			var h0v0:Number;
			var h0u1:Number;
			var h0v1:Number;
			var h1u0:Number;
			var h1v0:Number;
			var h1u1:Number;
			var h1v1:Number;
			
			var d:Number = 1 / imageDetail;
			for( var i:int=1; i<=imageDetail; i++ )
			{
				v = i * d;
				v1x0 = x0 + (x3 - x0) * v;
				v1y0 = y0 + (y3 - y0) * v;
				v1z0 = z0 + (z3 - z0) * v;
				v1x1 = x1 + (x2 - x1) * v;
				v1y1 = y1 + (y2 - y1) * v;
				v1z1 = z1 + (z2 - z1) * v;
				v1u0 = u0 + (u3 - u0) * v;
				v1v0 = v0 + (v3 - v0) * v;
				v1u1 = u1 + (u2 - u1) * v;
				v1v1 = v1 + (v2 - v1) * v;
				
				h0x0 = v0x0; h0y0 = v0y0; h0z0 = v0z0;
				h0x1 = v1x0; h0y1 = v1y0; h0z1 = v1z0;
				h0u0 = v0u0; h0v0 = v0v0;
				h0u1 = v1u0; h0v1 = v1v0;
				
				for ( var j:int = 1; j <=imageDetail; j++ )
				{
					u = j * d;
					h1x0 = v0x0 + (v0x1 - v0x0) * u;
					h1y0 = v0y0 + (v0y1 - v0y0) * u;
					h1z0 = v0z0 + (v0z1 - v0z0) * u;
					h1x1 = v1x0 + (v1x1 - v1x0) * u;
					h1y1 = v1y0 + (v1y1 - v1y0) * u;
					h1z1 = v1z0 + (v1z1 - v1z0) * u;
					h1u0 = v0u0 + (v0u1 - v0u0) * u;
					h1v0 = v0v0 + (v0v1 - v0v0) * u;
					h1u1 = v1u0 + (v1u1 - v1u0) * u;
					h1v1 = v1v0 + (v1v1 - v1v0) * u;
					
					polygon( h0x0, h0y0, h0z0,  h1x0, h1y0, h1z0,  h0x1, h0y1, h0z1,
							 h0u0, h0v0, h1u0, h1v0, h0u1, h0v1 );
					
					polygon( h1x0, h1y0, h1z0,  h1x1, h1y1, h1z1,  h0x1, h0y1, h0z1,
							 h1u0, h1v0, h1u1, h1v1, h0u1, h0v1 );
					
					h0x0 = h1x0; h0y0 = h1y0; h0z0 = h1z0;
					h0x1 = h1x1; h0y1 = h1y1; h0z1 = h1z1; 
					h0u0 = h1u0; h0v0 = h1v0;
					h0u1 = h1u1; h0v1 = h1v1;
				}
				
				v0x0 = v1x0; v0y0 = v1y0; v0z0 = v1z0;
				v0x1 = v1x1; v0y1 = v1y1; v0z1 = v1z1;
				v0u0 = v1u0; v0v0 = v1v0;
				v0u1 = v1u1; v0v1 = v1v1;
			}
		}
		
		/**
		 * 
		 * @param	vertices
		 * @param	indices
		 * @param	uvData
		 */
		public function drawTriangles( vertices:Array, indices:Array, uvData:Array=null ):void
		{			
			var i:int = 0;
			var len:int = indices.length;
			
			var i0:int;
			var i1:int;
			var i2:int;
			var i0y:int;
			var i0z:int;
			var i1y:int;
			var i1z:int;
			var i2y:int;
			var i2z:int;
			
			if ( __texture!=null )
			{
				var u0:int;
				var u1:int;
				var u2:int;
				var v0:int;
				var v1:int;
				var v2:int;
				for ( i = 0; i < len; i++ )
				{
					i0 = indices[i]; i++;
					i1 = indices[i]; i++;
					i2 = indices[i];
					u0 = i0 * 2;  v0 = u0 + 1;
					u1 = i1 * 2;  v1 = u1 + 1;
					u2 = i2 * 2;  v2 = u2 + 1;
					i0 *= 3;
					i1 *= 3;
					i2 *= 3;
					i0y = i0 + 1;  i0z = i0 + 2;
					i1y = i1 + 1;  i1z = i1 + 2;
					i2y = i2 + 1;  i2z = i2 + 2;
					polygon( vertices[i0], vertices[i0y], vertices[i0z],
							 vertices[i1], vertices[i1y], vertices[i1z],
							 vertices[i2], vertices[i2y], vertices[i2z],
							 uvData[u0], uvData[v0],
							 uvData[u1], uvData[v1],
							 uvData[u2], uvData[v2] );
				}
			}
			else
			{
				for ( i = 0; i < len; i++ )
				{
					i0 = 3 * indices[i]; i++;
					i1 = 3 * indices[i]; i++;
					i2 = 3 * indices[i];
					i0y = i0 + 1;  i0z = i0 + 2;
					i1y = i1 + 1;  i1z = i1 + 2;
					i2y = i2 + 1;  i2z = i2 + 2;
					polygonSolid( vertices[i0], vertices[i0y], vertices[i0z],
								  vertices[i1], vertices[i1y], vertices[i1z],
								  vertices[i2], vertices[i2y], vertices[i2z] );
				}
			}
		}
		
		/**
		 * 
		 * @param	vertices
		 * @param	faces
		 * @param	uvData
		 */
		public function drawMesh( vertices:Array, faces:Array, uvData:Array=null ):void
		{			
			var i:int = 0;
			var len:int = faces.length;
			
			var i0:int;
			var i1:int;
			var i2:int;
			var i0y:int;
			var i0z:int;
			var i1y:int;
			var i1z:int;
			var i2y:int;
			var i2z:int;
			
			if ( __texture!=null )
			{
				var u0:Number;
				var u1:Number;
				var u2:Number;
				var v0:Number;
				var v1:Number;
				var v2:Number;
				for ( i = 0; i < len; i++ )
				{
					i0 = faces[i]; 
					u0 = uvData[i * 2];
					v0 = uvData[i * 2 + 1];
					i++;
					i1 = faces[i]; 
					u1 = uvData[i * 2];
					v1 = uvData[i * 2 + 1];
					i++;
					i2 = faces[i];
					u2 = uvData[i * 2];
					v2 = uvData[i * 2 + 1];
					i0 *= 3;
					i1 *= 3;
					i2 *= 3;
					i0y = i0 + 1;  i0z = i0 + 2;
					i1y = i1 + 1;  i1z = i1 + 2;
					i2y = i2 + 1;  i2z = i2 + 2;
					polygon( vertices[i0], vertices[i0y], vertices[i0z],
							 vertices[i1], vertices[i1y], vertices[i1z],
							 vertices[i2], vertices[i2y], vertices[i2z],
							 u0, v0, u1, v1, u2, v2 );
				}
			}
			else
			{
				for ( i = 0; i < len; i++ )
				{
					i0 = 3 * faces[i]; i++;
					i1 = 3 * faces[i]; i++;
					i2 = 3 * faces[i];
					i0y = i0 + 1;  i0z = i0 + 2;
					i1y = i1 + 1;  i1z = i1 + 2;
					i2y = i2 + 1;  i2z = i2 + 2;
					polygonSolid( vertices[i0], vertices[i0y], vertices[i0z],
								  vertices[i1], vertices[i1y], vertices[i1z],
								  vertices[i2], vertices[i2y], vertices[i2z] );
				}
			}
		}
		
		//------------------------------------------------------------------------------------------------------------------- 
		// render
		//-------------------------------------------------------------------------------------------------------------------
		
		/**
		 * @private
		 */
		private function render():void
		{
			// add last buffer
			terminateShapePath();
			
			//
			TASK_DAT.sortOn("za", Array.DESCENDING | Array.NUMERIC );
			
			//
			var stk_do:Boolean = true;
			var stk_c:uint     = 0;
			var stk_a:Number   = -1;
			
			var _task:RenderTaskObject;
			var task:RenderTask;
			
			var cmd_start:int;
			var cmd_num:int;
			var c:int;
			var xi:int;
			var yi:int;
			
			//
			gc.lineStyle();
			var tmp_stroke_index:uint = 0;
			var stroke_index:uint;
			
			for ( var i:int = 0; i < task_count; i++ )
			{	
				_task = RenderTaskObject( TASK_DAT[i] );
				switch( _task.kind )
				{
					case TASK_SHAPE:
						task = RenderTask(_task);
						cmd_start = task.command_start;
						cmd_num   = task.command_num;
						xi = task.path_start;
						yi = xi + 1;
						stroke_index = task.stroke_index;
						if ( tmp_stroke_index != stroke_index )
						{
							IStrokeTask( STROKE_TASK[stroke_index] ).setLineStyle(gc);
							tmp_stroke_index = stroke_index;
						}
						
						gc.moveTo( 0, 0 );
						task.applyFill( gc );
						var cmd:int;
						
						gc.moveTo( paths[xi], paths[yi]);
						xi += 3;
						yi += 3;
						for ( c=1; c<cmd_num; c++ )
						{
							cmd_start++;
							switch( commands[cmd_start] )
							{
								case 1:
									gc.moveTo( paths[xi], paths[yi] );
									xi += 3;
									yi += 3;
									break;
								case 2:
									gc.lineTo( paths[xi], paths[yi] );
									xi += 3;
									yi += 3;
									break;
								case 3:
									gc.curveTo( paths[xi], paths[yi], paths[xi+3], paths[yi+3] );
									xi += 6;
									yi += 6;
									break;
								default:
									break;
							}
						}
						gc.endFill();
						break;
						
					case TASK_STROKE:
						task = RenderTask(_task);
						cmd_start = task.command_start;
						cmd_num   = task.command_num;
						xi = task.path_start;
						yi = xi + 1;
						stroke_index = task.stroke_index;
						if ( tmp_stroke_index != stroke_index )
						{
							IStrokeTask( STROKE_TASK[stroke_index] ).setLineStyle(gc);
							tmp_stroke_index = stroke_index;
						}
						
						var zi:int = xi + 2;
						var skip:Boolean = false;
						
						if ( paths[zi]>0 )
							gc.moveTo( paths[xi], paths[yi] );
						else
							skip = true;
						xi += 3;
						yi += 3;
						zi += 3;
						for ( c=1; c<cmd_num; c++ )
						{
							cmd_start++;
							if ( commands[cmd_start] == 2 )
							{
								if (  paths[zi]<=0 )
								{
									skip = true;
								}
								else if ( skip )
								{
									gc.moveTo( paths[xi], paths[yi] );
									skip = false;
								}
								else
								{
									gc.lineTo( paths[xi], paths[yi] );
								}
								xi += 3;
								yi += 3;
								zi += 3;
							}
							else
							{
								if (  paths[zi]<=0 || paths[zi+3]<=0  )
								{
									skip = true;
								}
								else if ( skip )
								{
									gc.moveTo( paths[xi+3], paths[yi+3] );
									skip = false;
								}
								else
								{
									gc.curveTo( paths[xi], paths[yi], paths[xi+3], paths[yi+3] );
								}
								xi += 6; 
								yi += 6;
								zi += 6;
							}
						}
						break;
						
					case TASK_POLYGON:
						stroke_index = _task.stroke_index;
						if ( tmp_stroke_index != stroke_index )
						{
							IStrokeTask( STROKE_TASK[stroke_index] ).setLineStyle(gc);
							tmp_stroke_index = stroke_index;
						}
						gc.moveTo( 0, 0 );
						PolygonTextureTask(_task).render( this );
						break;
						
					case TASK_POLYGON_SOLID:
						stroke_index = _task.stroke_index;
						if ( tmp_stroke_index != stroke_index )
						{
							IStrokeTask( STROKE_TASK[stroke_index] ).setLineStyle(gc);
							tmp_stroke_index = stroke_index;
						}
						gc.moveTo( 0, 0 );
						IGraphicsTask(_task).render( gc );
						break;
						
					case TASK_IMAGE:
						stroke_index = _task.stroke_index;
						if ( tmp_stroke_index != stroke_index )
						{
							IStrokeTask( STROKE_TASK[stroke_index] ).setLineStyle(gc);
							tmp_stroke_index = stroke_index;
						}
						gc.moveTo( 0, 0 );
						ImageTask(_task).render( this );
						break;
						
					case TASK_POINT:
						if ( tmp_stroke_index != 0 )
						{
							gc.lineStyle();
							tmp_stroke_index = 0;
						}
						gc.moveTo( 0, 0 );
						IGraphicsTask(_task).render( gc );
						break;
						
					case TASK_PIXEL:
						PixelTask(_task).render( pixelbitmap );
						break;
							
					default:
						break;
				}
			}
		}
		
	}
	
}